// Aquastream USB Interface
// Erzeugt von unwissender
// Inspiriert von Christian Unger's aeinfo

#include <sys/stat.h>
#include <fcntl.h>


extern int VERBOSE;
#undef PINT

#include <windows.h>
#include <setupapi.h>
#include <iostream>
#include <iomanip>
#include <memory>

typedef struct _HIDD_ATTRIBUTES {
  ULONG  Size;
  USHORT  VendorID;
  USHORT  ProductID;
  USHORT  VersionNumber;
} HIDD_ATTRIBUTES, *PHIDD_ATTRIBUTES;


typedef struct _HIDP_PREPARSED_DATA * PHIDP_PREPARSED_DATA;
typedef struct _HIDP_CAPS {
  USHORT  Usage;
  USHORT  UsagePage;
  USHORT  InputReportByteLength;
  USHORT  OutputReportByteLength;
  USHORT  FeatureReportByteLength;
  USHORT  Reserved[17];
  USHORT  NumberLinkCollectionNodes;
  USHORT  NumberInputButtonCaps;
  USHORT  NumberInputValueCaps;
  USHORT  NumberInputDataIndices;
  USHORT  NumberOutputButtonCaps;
  USHORT  NumberOutputValueCaps;
  USHORT  NumberOutputDataIndices;
  USHORT  NumberFeatureButtonCaps;
  USHORT  NumberFeatureValueCaps;
  USHORT  NumberFeatureDataIndices;
} HIDP_CAPS, *PHIDP_CAPS;


// Klasse zur allgemeinen Kommunikation mit einem HID-Device (Lesen von FeatureReports)
class USBCommunication
{
  HINSTANCE hidDll;
  HANDLE DeviceHandle;
  
  VOID     (_stdcall *HidD_GetHidGuid)       (OUT LPGUID HidGuid);
  BOOLEAN  (_stdcall *HidD_GetAttributes)    (IN HANDLE  HidDeviceObject, OUT PHIDD_ATTRIBUTES  Attributes);
  BOOLEAN  (_stdcall *HidD_GetPreparsedData) (IN HANDLE  HidDeviceObject, OUT PHIDP_PREPARSED_DATA  *PreparsedData);
  NTSTATUS (_stdcall *HidP_GetCaps)          (IN PHIDP_PREPARSED_DATA  PreparsedData, OUT PHIDP_CAPS  Capabilities);
  BOOLEAN  (_stdcall *HidD_FreePreparsedData)(IN PHIDP_PREPARSED_DATA  PreparsedData);
  BOOLEAN  (_stdcall *HidD_GetFeature)       (IN HANDLE  HidDeviceObject, OUT PVOID  ReportBuffer, IN ULONG  ReportBufferLength);
  ULONG    (_stdcall *HidD_Hello)            (PCHAR Buffer, ULONG BufferLength);

  unsigned long FeatureLength;
public:
  USBCommunication ( unsigned long VID, unsigned long PID ) 
  {
    hidDll = LoadLibrary("hid.dll");
    if( hidDll == NULL )
    {
      #ifdef _WITH_CONIO_
        if(VERBOSE) std::cout << "hid.dll not found" << std::endl;
      #endif
      throw "hid.dll not found or loadable!";
    }
    else
    {
      #ifdef _WITH_CONIO_
        if(VERBOSE) std::cout << "extracting entry-points" << std::endl;
      #endif
      HidD_GetHidGuid = (VOID (_stdcall *)(OUT LPGUID))GetProcAddress(hidDll,"HidD_GetHidGuid");
      HidD_GetAttributes = (BOOLEAN (_stdcall *) (HANDLE,PHIDD_ATTRIBUTES))GetProcAddress(hidDll,"HidD_GetAttributes");
      HidD_Hello = (ULONG (_stdcall *) (OUT PCHAR,IN ULONG))GetProcAddress(hidDll,"HidD_Hello");
      HidD_GetPreparsedData = (BOOLEAN (_stdcall *) (HANDLE,PHIDP_PREPARSED_DATA*))GetProcAddress(hidDll,"HidD_GetPreparsedData");
      HidP_GetCaps = (NTSTATUS (_stdcall *) (PHIDP_PREPARSED_DATA,PHIDP_CAPS))GetProcAddress(hidDll,"HidP_GetCaps");
      HidD_FreePreparsedData= (BOOLEAN (_stdcall *) (PHIDP_PREPARSED_DATA))GetProcAddress(hidDll,"HidD_FreePreparsedData");
      HidD_GetFeature = (BOOLEAN (_stdcall *) (HANDLE,PVOID,ULONG))GetProcAddress(hidDll,"HidD_GetFeature");

      #ifdef _WITH_CONIO_
        if(VERBOSE) {
          char tmp[10];
          tmp[9] = '\0';
          std::cout << "testing hidhello" << std::endl;
          HidD_Hello((PCHAR)tmp,10);
          std::cout << tmp << std::endl;
        }
      #endif
    }
    DeviceHandle = NULL;
    FeatureLength = 0;


    // this source-code is copy'n'pasted from sources found at www.lvr.com
    GUID HidGuid;

    HidD_GetHidGuid(&HidGuid);
    
    HANDLE hDevInfo=SetupDiGetClassDevs(&HidGuid, NULL, NULL, DIGCF_PRESENT|DIGCF_INTERFACEDEVICE);
    #ifdef _WITH_CONIO_
      if(VERBOSE) std::cout << "2" << std::endl;
    #endif
    
    SP_DEVICE_INTERFACE_DATA devInfoData;
    devInfoData.cbSize = sizeof(devInfoData);
    int MemberIndex = 0;
    bool LastDevice = false;
    unsigned long Length;
    unsigned long Required;
    HIDD_ATTRIBUTES Attributes;
    bool MyDeviceDetected = false;
    
    do
      {
        LONG Result=SetupDiEnumDeviceInterfaces(hDevInfo, 0, &HidGuid, MemberIndex, &devInfoData);
        #ifdef _WITH_CONIO_
          if(VERBOSE) std::cout << "3" << std::endl;
        #endif
        
        if (Result != 0)
        {
            Result = SetupDiGetDeviceInterfaceDetail(hDevInfo, &devInfoData, NULL, 0, &Length, NULL);
            #ifdef _WITH_CONIO_
              if(VERBOSE) std::cout << "4" << std::endl;
            #endif
            
            PSP_DEVICE_INTERFACE_DETAIL_DATA detailData = (PSP_DEVICE_INTERFACE_DETAIL_DATA)malloc(Length);
            
            //Set cbSize in the detailData structure.
            detailData -> cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);
            
            //Call the function again, this time passing it the returned buffer size.
            Result = SetupDiGetDeviceInterfaceDetail(hDevInfo, &devInfoData, detailData, Length, &Required, NULL);
            #ifdef _WITH_CONIO_
              if(VERBOSE) std::cout << "5" << std::endl;
            #endif

            DeviceHandle = CreateFile(detailData->DevicePath, GENERIC_READ|GENERIC_WRITE, FILE_SHARE_READ|FILE_SHARE_WRITE, (LPSECURITY_ATTRIBUTES)NULL, OPEN_EXISTING, 0, NULL);
            #ifdef _WITH_CONIO_
              if(VERBOSE) std::cout  << "6" << std::endl;
            #endif
            
            //DisplayLastError("CreateFile: ");
            Attributes.Size = sizeof(Attributes);
            
            Result = HidD_GetAttributes(DeviceHandle, &Attributes);
            #ifdef _WITH_CONIO_
              if(VERBOSE) std::cout << "7" << std::endl;
            #endif
            
            //DisplayLastError("HidD_GetAttributes: ");
            
            //Is it the desired device?
            if (Attributes.VendorID == VID && Attributes.ProductID == PID)
            {
              //MyDevicePathName = detailData->DevicePath;
              // Get a handle for writing Output reports.
              MyDeviceDetected = true;
              
              #ifdef _WITH_CONIO_
                if(VERBOSE) std::cout << "found the device!!" << Attributes.VersionNumber  << std::endl;
              #endif
              
            }

            else
              //The Product ID doesn't match.
              CloseHandle(DeviceHandle);
            
            free(detailData);
          }  //if (Result != 0)
        else
          //SetupDiEnumDeviceInterfaces returned 0, so there are no more devices to check.
          LastDevice = true;

        //If we haven't found the device yet, and haven't tried every available device,
        //try the next one.
        
        MemberIndex = MemberIndex + 1;
        
      } //do
    
    while (!LastDevice && !MyDeviceDetected);
    SetupDiDestroyDeviceInfoList(hDevInfo);

    if (!MyDeviceDetected)
      throw "Device not found";

    FeatureLength = 0;
    PHIDP_PREPARSED_DATA data;
    if (HidD_GetPreparsedData(DeviceHandle,&data))
    {
      HIDP_CAPS  caps;
      NTSTATUS r = HidP_GetCaps(data,(PHIDP_CAPS)&caps);
      if (r==0x110000 )
      {
        #ifdef _WITH_CONIO_
          if (VERBOSE) {
            std::cout << "Capabilities:\n";
            std::cout << "\n  Usage: " << caps.Usage;
            std::cout << "\n  UsagePage: " << caps.UsagePage;
            std::cout << "\n  InputReportByteLength: " << caps.InputReportByteLength;
            std::cout << "\n  OutputReportByteLength: " << caps.OutputReportByteLength;
            std::cout << "\n  FeatureReportByteLength: " << caps.FeatureReportByteLength;
            std::cout << "\n  NumberLinkCollectionNodes: " << caps.NumberLinkCollectionNodes;
            std::cout << "\n  NumberInputButtonCaps: " << caps.NumberInputButtonCaps;
            std::cout << "\n  NumberInputValueCaps: " << caps.NumberInputValueCaps;
            std::cout << "\n  NumberInputDataIndices: " << caps.NumberInputDataIndices;
            std::cout << "\n  NumberOutputButtonCaps: " << caps.NumberOutputButtonCaps;
            std::cout << "\n  NumberOutputValueCaps: " << caps.NumberOutputValueCaps;
            std::cout << "\n  NumberOutputDataIndices: " << caps.NumberOutputDataIndices;
            std::cout << "\n  NumberFeatureButtonCaps: " << caps.NumberFeatureButtonCaps;
            std::cout << "\n  NumberFeatureValueCaps: " << caps.NumberFeatureValueCaps;
            std::cout << "\n  NumberFeatureDataIndices: " << caps.NumberFeatureDataIndices << std::endl;
          }
        #endif

        FeatureLength = caps.FeatureReportByteLength;
      }
      HidD_FreePreparsedData(data);
    }

    if (FeatureLength == 0)
      throw "Unknown Interface to device/no FeatureReports";

  }

  virtual ~USBCommunication () 
  {
    CloseHandle(DeviceHandle);
    FreeLibrary(hidDll);
  }


  unsigned char *readFeature(unsigned char Feature, unsigned char *pre_allocated_buffer = 0)
  {
    unsigned char *buffer;
    if (!pre_allocated_buffer)
      buffer = new unsigned char[FeatureLength+1];
    else
      buffer = pre_allocated_buffer;
    
    buffer[0] = Feature;
    BOOLEAN Result = HidD_GetFeature( DeviceHandle, buffer, FeatureLength);

    if (Result)
    {
      #ifdef _WITH_CONIO_
        if (VERBOSE)
          handle_buffer(buffer);
      #endif
      return buffer;
    }

    if (!pre_allocated_buffer)
      delete [] buffer;

    return 0;
  }
  
private:
  #ifdef _WITH_CONIO_
    void debug_buffer_hex ( unsigned char *buffer, int len ) 
    {
      int i;
    
      printf( "hex-dump of recv'd buffer:\n" );
      printf( "---------------------------\n" );
    
      for( i = 0; i < len; ) {
        unsigned char c = (unsigned char)buffer[i];
        printf( "%02X ", (int)c );
        ++i;
      
        if( (i % 20) == 0 ) printf( "\n" );
      }
      printf( "\n" "---------------------------" "\n" );
    }


    void debug_buffer ( unsigned char *buffer, int len ) 
    {
      int i;
      char tmp[2];
      tmp[0] = ' ';
      tmp[1] = '\0';
    
      printf( "buffer:>>>" );
      for( i = 0; i < len; ++i ) {
        char c = buffer[i];
      
        if( c >= ' ' && c <= 'z' ) {
          tmp[0] = c;
        }else{
          tmp[0] = '*';
        }
      
        printf( "%s", (char*)tmp );
      }
      printf( "<<<" "\n" );
    }


    void handle_buffer ( unsigned char *buffer ) 
    {
      debug_buffer_hex( buffer, FeatureLength+1 );
      debug_buffer( buffer, FeatureLength+1 );
    }
  #endif
};



static const double SCALE12V = 61.00;			//Skalierungsfaktor 12V
static const double SCALEFANOUT = 63.00;		//Skalierungsfaktor Fanout
static const double SCALEPUMPCURRENT = 1.6;	//Skalierungsfaktor Pumpenstrom
static const int F_CPU = 12000000;				//CPU clock of aquastream xt
static const int F_TIMER_PUMP = F_CPU / 8;		//Timer count frequency (base pump frequency)
static const int TEMP_SCALER = 100;				//Scale of Temperature
static const int CONTROLLER_TIMER = 100;		//Controller cycle time 100ms
static const double LIMITER_SCALE = 0.01333155;	//Frequency Limiter scale
static const double F_TIMER_RPM = 46875.0;		//Scale for Fan RPM
static const long MIN_RPM = 300000;				//Minimum Fan RPM
static const long MIN_FLOW = 600000;				//Minimum Fan RPM


class Aquastream
{
#pragma pack( push )
#pragma pack(1)
  typedef struct
  {
    unsigned char  ReportID;
    unsigned short raw_data[6];
    unsigned short tempPump;
    unsigned short tempExt;
    unsigned short tempWater;
    unsigned short frequency;
    unsigned short max_frequency;
    unsigned long  flow;
    unsigned long  fanRPM;
    unsigned char  fanPWR;
    unsigned char  alarm;
    unsigned char  mode;
    unsigned long  controllerOut;
      signed long  controllerI;
      signed long  controllerP;
      signed long  controllerD;
    unsigned short firmware;
    unsigned short bootloader;
    unsigned short hardware;
    unsigned short dummy;
    unsigned short serial;
    unsigned char  publicKey[6];

    double GetWaterTemperature() const    { return static_cast<double>(tempWater)/TEMP_SCALER; }
    double GetExternalTemperature() const { return static_cast<double>(tempExt)/TEMP_SCALER; }
    double GetPumpTemperature() const     { return static_cast<double>(tempPump)/TEMP_SCALER; }
    double GetFanVoltage() const          { return static_cast<double>(raw_data[3])/SCALEFANOUT; }
    double GetFanPercentage() const       { return static_cast<double>(fanPWR)/2.55; }
    int    GetFanRotation(double fanDiv) const {
      if (fanRPM) 
        return static_cast<int>( F_TIMER_RPM*60.0 * fanDiv / ( static_cast<double>(fanRPM)*4.0 )); 
      else 
        return 0; 
    };
    unsigned char GetAlarmBits() const {
      if (mode&4)
        return alarm;       // Ultra: Alle Bits relevant
      else
        return alarm&0xe4;  // Standard/Advanced: Bits fr Flow,Fan,Wassertemp,ExtTemp lschen
    }
  } pumpData;

  typedef struct
  {
    unsigned char  ReportID;
    unsigned char  i2cAddress;
    unsigned char  i2cSettings;
    unsigned char  pumpModeA;
    unsigned char  pumpModeB;
    unsigned char  sensorBridge;
    unsigned char  measureFanEdges;
    unsigned char  measureFlowEdges;
    unsigned short pumpFrequency;
    unsigned long  frequencyResetCycle;
    unsigned char  alarmEnable;
    unsigned char  tachoSettings;
    unsigned short tachoFrequency;
    unsigned long  flowAlarmValue;
    unsigned short sensorAlarmTemperature[2];
    unsigned char  fanModeSettings;
    unsigned char  fanManualPower;
    unsigned short controllerHysterese;
    unsigned char  controllerSensor;
    unsigned short controllerSetTemp;
    unsigned short controllerP;
    unsigned short controllerI;
    unsigned short controllerD;
    unsigned short sensorMinTemperature;
    unsigned short sensorMaxTemperature;
    unsigned char  fanMinimumPower;
    unsigned char  fanMaximumPower;
    unsigned char  ledSettings;
    unsigned char  aquabusTimeout;
    unsigned short minPumpFreqency;
    unsigned short maxPumpFreqency;

    unsigned char GetAlarmMask() const { return alarmEnable; }
    unsigned char GetFanDiv() const    { return measureFanEdges; }
  } pumpSettings;
#pragma pack(pop)

  #define VID_AQUACOMPUTER 0x0c70
  #define PID_AQUASTREAM   0xf0b6
  unsigned char *data, *settings;

public:
  Aquastream() : data(0), settings(0)
  {
  }

  ~Aquastream()
  {
    clear();
  }

  bool read()   // Bei true wurden Features erfolgreich gelesen
  {
    if (!usb.get())
    {
      try {
        clear();
        usb = std::auto_ptr<USBCommunication>(new USBCommunication(VID_AQUACOMPUTER, PID_AQUASTREAM));
        OutputDebugString("Aquastream connected");
      }
      catch (char *fault) {
        OutputDebugString(fault);
        return false;
      }
    }
    unsigned char *tmpdata     = usb->readFeature(4, data);
    unsigned char *tmpsettings = usb->readFeature(6, settings);

    if (tmpdata && tmpsettings)
    {
      // Lesevorgang erfolgreich
      data = tmpdata;
      settings = tmpsettings;
      return true;
    }
    else
    {
      OutputDebugString("Error Reading Feature from Aquastream... Restart");
      usb.reset();
      clear();
      return false;
    }
  }
  bool hasData() const
  {
    return data && settings;
  }

  double GetWaterTemperature() const    { return reinterpret_cast<const pumpData*>(data)->GetWaterTemperature(); }
  double GetExternalTemperature() const { return reinterpret_cast<const pumpData*>(data)->GetExternalTemperature(); }
  double GetPumpTemperature() const     { return reinterpret_cast<const pumpData*>(data)->GetPumpTemperature(); }
  double GetFanVoltage() const          { return reinterpret_cast<const pumpData*>(data)->GetFanVoltage(); }
  double GetFanPercentage() const       { return reinterpret_cast<const pumpData*>(data)->GetFanPercentage(); }
  int    GetFanRotation() const         { return reinterpret_cast<const pumpData*>(data)->GetFanRotation(reinterpret_cast<const pumpSettings*>(settings)->GetFanDiv()); }
  bool   GetAlarm() const               { return (reinterpret_cast<const pumpData*>(data)->GetAlarmBits() & reinterpret_cast<const pumpSettings*>(settings)->GetAlarmMask()) != 0; };

private:
  void clear()
  {
    delete [] data;
    delete [] settings;
    data = 0;
    settings = 0;
  }

  std::auto_ptr<USBCommunication> usb;
};
